/* ------------------------------------------------------------------------------
 *
 *  # D3.js - gradient encoding line chart
 *
 *  Demo d3.js gradient encoding line chart setup with .tsv data source
 *
 * ---------------------------------------------------------------------------- */


// Setup module
// ------------------------------

var D3LineGradient = function() {


    //
    // Setup module components
    //

    // Chart
    var _lineGradient = function() {
        if (typeof d3 == 'undefined') {
            console.warn('Warning - d3.min.js is not loaded.');
            return;
        }

        // Main variables
        var element = document.getElementById('d3-line-gradient'),
            height = 400;


        // Initialize chart only if element exsists in the DOM
        if(element) {

            // Basic setup
            // ------------------------------

            // Define main variables
            var d3Container = d3.select(element),
                margin = {top: 5, right: 20, bottom: 20, left: 40},
                width = d3Container.node().getBoundingClientRect().width - margin.left - margin.right,
                height = height - margin.top - margin.bottom - 5;

            // Format data
            var parseDate = d3.time.format("%Y%m%d").parse;



            // Construct scales
            // ------------------------------

            // Horizontal
            var x = d3.time.scale()
                .range([0, width]);

            // Vertical
            var y = d3.scale.linear()
                .range([height, 0]);



            // Create axes
            // ------------------------------

            // Horizontal
            var xAxis = d3.svg.axis()
                .scale(x)
                .orient("bottom")
                .ticks(7)
                .tickFormat(d3.time.format("%b"));

            // Vertical
            var yAxis = d3.svg.axis()
                .scale(y)
                .orient("left");



            // Create chart
            // ------------------------------

            // Add SVG element
            var container = d3Container.append("svg");

            // Add SVG group
            var svg = container
                .attr("width", width + margin.left + margin.right)
                .attr("height", height + margin.top + margin.bottom)
                .append("g")
                    .attr("transform", "translate(" + margin.left + "," + margin.top + ")");


            // Construct chart layout
            // ------------------------------

            // Line
            var line = d3.svg.line()
                .interpolate("basis")
                .x(function(d) { return x(d.date); })
                .y(function(d) { return y(d.temperature); });


            // Load data
            // ------------------------------

            d3.tsv("http://demo.interface.club/limitless/demo/bs4/Template/global_assets/demo_data/d3/lines/lines_gradient.tsv", function(error, data) {

                // Pull out values
                data.forEach(function(d) {
                    d.date = parseDate(d.date);
                    d.temperature = +d.temperature;
                });


                // Set input domains
                // ------------------------------

                // Horizontal
                x.domain([data[0].date, data[data.length - 1].date]);

                // Vertical
                y.domain(d3.extent(data, function(d) { return d.temperature; }));


                //
                // Append chart elements
                //

                // Add gradient
                svg.append("linearGradient")
                    .attr("id", "temperature-gradient")
                    .attr("gradientUnits", "userSpaceOnUse")
                    .attr("x1", 0)
                    .attr("y1", y(50))
                    .attr("x2", 0)
                    .attr("y2", y(60))
                    .selectAll("stop")
                    .data([
                        {offset: "0%", color: "#4CAF50"},
                        {offset: "50%", color: "#81C784"},
                        {offset: "100%", color: "#FF5722"}
                    ])
                    .enter()
                    .append("stop")
                        .attr("offset", function(d) { return d.offset; })
                        .attr("stop-color", function(d) { return d.color; });

                // Add line
                svg.append("path")
                    .datum(data)
                    .attr("class", "d3-line d3-line-medium")
                    .attr("fill", "none")
                    .attr("stroke", "url(#temperature-gradient)")
                    .attr("stroke-width", 2)
                    .attr("d", line);



                // Append axes
                // ------------------------------

                // Horizontal
                svg.append("g")
                    .attr("class", "d3-axis d3-axis-horizontal d3-axis-strong")
                    .attr("transform", "translate(0," + height + ")")
                    .call(xAxis);

                // Vertical
                var verticalAxis = svg.append("g")
                    .attr("class", "d3-axis d3-axis-vertical d3-axis-strong")
                    .call(yAxis);

                // Add text label
                verticalAxis.append("text")
                    .attr("transform", "rotate(-90)")
                    .attr("y", 10)
                    .attr("dy", ".71em")
                    .style("text-anchor", "end")
                    .style("fill", "#999")
                    .style("font-size", 12)
                    .text("Temperature (ºF)");
            });



            // Resize chart
            // ------------------------------

            // Call function on window resize
            $(window).on('resize', resize);

            // Call function on sidebar width change
            $('.sidebar-control').on('click', resize);

            // Resize function
            // 
            // Since D3 doesn't support SVG resize by default,
            // we need to manually specify parts of the graph that need to 
            // be updated on window resize
            function resize() {

                // Layout variables
                width = d3Container.node().getBoundingClientRect().width - margin.left - margin.right;


                // Layout
                // -------------------------

                // Main svg width
                container.attr("width", width + margin.left + margin.right);

                // Width of appended group
                svg.attr("width", width + margin.left + margin.right);


                // Axes
                // -------------------------

                // Horizontal range
                x.range([0, width]);

                // Horizontal axis
                svg.selectAll('.d3-axis-horizontal').call(xAxis);


                // Chart elements
                // -------------------------

                // Line path
                svg.selectAll('.d3-line').attr("d", line);
            }
        }
    };


    //
    // Return objects assigned to module
    //

    return {
        init: function() {
            _lineGradient();
        }
    }
}();


// Initialize module
// ------------------------------

document.addEventListener('DOMContentLoaded', function() {
    D3LineGradient.init();
});
